<?php
// pages/transactions.php
include '../config/database.php';
include '../includes/header.php';

// Generate transaction code
$transaction_code = "INV" . date('YmdHis');

// Process transaction
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $customer_id = escape($_POST['customer_id'] ?? '');
    $payment_method = escape($_POST['payment_method']);
    $payment_amount = escape($_POST['payment_amount']);
    $notes = escape($_POST['notes'] ?? '');
    $discount_amount = escape($_POST['discount_amount'] ?? 0);
    
    $cart_items = json_decode($_POST['cart_items'], true) ?? [];
    
    if (empty($cart_items)) {
        $error = "Keranjang belanja kosong!";
    } else {
        $total_amount = array_sum(array_column($cart_items, 'subtotal'));
        $final_amount = $total_amount - $discount_amount;
        $change_amount = $payment_method === 'cash' ? $payment_amount - $final_amount : 0;
        
        if ($payment_method === 'cash' && $change_amount < 0) {
            $error = "Jumlah pembayaran kurang!";
        } else {
            $sql = "INSERT INTO transactions (transaction_code, customer_id, user_id, total_amount, discount_amount, final_amount, payment_method, payment_amount, change_amount, notes) 
                    VALUES ('$transaction_code', " . ($customer_id ? "'$customer_id'" : "NULL") . ", '{$_SESSION['user_id']}', '$total_amount', '$discount_amount', '$final_amount', '$payment_method', '$payment_amount', '$change_amount', '$notes')";
            
            if (query($sql)) {
                $transaction_id = lastInsertId();
                
                foreach ($cart_items as $item) {
                    $product_id = escape($item['productId']);
                    $quantity = escape($item['quantity']);
                    $price = escape($item['price']);
                    $subtotal = escape($item['subtotal']);
                    $product_type = escape($item['type']);
                    $product_name = escape($item['name']);
                    
                    $item_sql = "INSERT INTO transaction_items (transaction_id, product_id, product_type, product_name, product_price, quantity, subtotal) 
                                 VALUES ('$transaction_id', '$product_id', '$product_type', '$product_name', '$price', '$quantity', '$subtotal')";
                    query($item_sql);
                    
                    if ($product_type !== 'service') {
                        query("UPDATE products SET stock = stock - $quantity WHERE id = '$product_id'");
                    }
                }
                
                // SIMPAN DATA SUCCESS DI SESSION TANPA REDIRECT
                $_SESSION['success_data'] = json_encode([
                    'code' => $transaction_code,
                    'total' => $final_amount,
                    'payment' => $payment_amount,
                    'change' => $change_amount,
                    'discount' => $discount_amount,
                    'items' => $cart_items
                ]);

                if (isset($_SESSION['cart'])) {
                    unset($_SESSION['cart']);
                }

                // JANGAN REDIRECT - biarkan halaman reload secara normal
                // echo "<script>window.location.href = 'transactions.php?success=1';</script>";
                // exit();
                
                // Set flag untuk menunjukkan transaksi berhasil
                $transaction_success = true;
            } else {
                $error = "Gagal memproses transaksi! Error: " . mysqli_error($connection);
            }
        }
    }
}

// Get customers
$customers = query("SELECT * FROM customers WHERE is_active = 1 ORDER BY name");

// Check for success message - MODIFIKASI INI
$success_data = null;
if (isset($_SESSION['success_data'])) {
    $success_data = json_decode($_SESSION['success_data'], true);
    unset($_SESSION['success_data']); // Hapus setelah digunakan
}
?>

<div class="row">
    <div class="col-12">
        <div class="card mb-4 shadow-sm border-0">
            <div class="card-body">
                <div class="row align-items-center">
                    <div class="col-md-4">
                        <h4 class="mb-0"><i class="fas fa-cash-register me-2 text-primary"></i>Transaksi Baru</h4>
                    </div>
                    <div class="col-md-4 text-center">
                        <div id="clockDisplay" class="text-muted">
                            <i class="fas fa-clock me-2"></i><span id="currentTime"></span>
                        </div>
                    </div>
                    <div class="col-md-4 text-md-end">
                        <div class="transaction-code">
                            <span class="badge bg-primary bg-gradient fs-6 px-3 py-2"><?php echo $transaction_code; ?></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php if (isset($error)): ?>
<div class="alert alert-danger alert-dismissible fade show shadow-sm">
    <i class="fas fa-exclamation-circle me-2"></i><?php echo $error; ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<div class="row">
    <!-- Product List -->
    <div class="col-lg-4 mb-4">
        <div class="card h-100 shadow-sm border-0">
            <div class="card-header bg-gradient-primary text-white">
                <h5 class="mb-0"><i class="fas fa-boxes me-2"></i>Daftar Produk</h5>
            </div>
            <div class="card-body p-0">
                <div class="p-3 bg-light border-bottom">
                    <div class="input-group">
                        <span class="input-group-text bg-white border-end-0">
                            <i class="fas fa-search text-muted"></i>
                        </span>
                        <input type="text" id="searchProduct" class="form-control border-start-0 ps-0" 
                               placeholder="Cari produk..." autofocus>
                        <button class="btn btn-outline-secondary" type="button" id="scanBarcodeBtn">
                            <i class="fas fa-barcode"></i>
                        </button>
                    </div>
                    <div class="mt-2">
                        <select id="categoryFilter" class="form-select form-select-sm">
                            <option value="">Semua Kategori</option>
                        </select>
                    </div>
                </div>
                <div id="productList" class="p-3" style="max-height: 550px; overflow-y: auto;">
                    <div class="text-center text-muted py-4">
                        <div class="spinner-border spinner-border-sm text-primary me-2" role="status"></div>
                        <span>Memuat produk...</span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Shopping Cart -->
    <div class="col-lg-8">
        <div class="card h-100 shadow-sm border-0">
            <div class="card-header bg-gradient-success text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-shopping-cart me-2"></i>Keranjang Belanja</h5>
                    <span id="cartCount" class="badge bg-light text-dark">0 items</span>
                </div>
            </div>
            <div class="card-body">
                <form method="POST" id="transactionForm">
                    <input type="hidden" name="cart_items" id="cartItemsInput">
                    <input type="hidden" name="discount_amount" id="discountAmountInput" value="0">
                    
                    <div class="row g-3 mb-3">
                        <div class="col-md-6">
                            <label for="customer_id" class="form-label">
                                <i class="fas fa-user me-1"></i>Pelanggan
                            </label>
                            <div class="input-group">
                                <select class="form-select" id="customer_id" name="customer_id">
                                    <option value="">Walk-in Customer</option>
                                    <?php while ($customer = $customers->fetch_assoc()): ?>
                                    <option value="<?php echo $customer['id']; ?>">
                                        <?php echo $customer['name']; ?> - <?php echo $customer['phone']; ?>
                                    </option>
                                    <?php endwhile; ?>
                                </select>
                                <button type="button" class="btn btn-outline-secondary" data-bs-toggle="modal" data-bs-target="#addCustomerModal">
                                    <i class="fas fa-plus"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <label for="payment_method" class="form-label">
                                <i class="fas fa-credit-card me-1"></i>Metode Bayar
                            </label>
                            <select class="form-select" id="payment_method" name="payment_method" required>
                                <option value="cash">💵 Tunai</option>
                                <option value="transfer">🏦 Transfer Bank</option>
                                <option value="debit_card">💳 Kartu Debit</option>
                                <option value="credit_card">💳 Kartu Kredit</option>
                                <option value="e_wallet">📱 E-Wallet</option>
                            </select>
                        </div>
                    </div>

                    <!-- Cart Items -->
                    <div id="cartContainer" class="mb-3" style="max-height: 300px; overflow-y: auto;">
                        <div class="text-center text-muted py-5">
                            <i class="fas fa-shopping-cart fa-4x mb-3 opacity-25"></i>
                            <p class="mb-0">Keranjang kosong</p>
                            <small>Tambahkan produk dari daftar di samping</small>
                        </div>
                    </div>

                    <!-- Quick Actions -->
                    <div class="mb-3">
                        <div class="btn-group w-100" role="group">
                            <button type="button" class="btn btn-outline-danger btn-sm" onclick="clearCart()">
                                <i class="fas fa-trash me-1"></i>Hapus Semua
                            </button>
                            <button type="button" class="btn btn-outline-info btn-sm" onclick="holdTransaction()">
                                <i class="fas fa-pause me-1"></i>Tunda
                            </button>
                            <button type="button" class="btn btn-outline-warning btn-sm" onclick="applyDiscount()">
                                <i class="fas fa-percent me-1"></i>Diskon
                            </button>
                        </div>
                    </div>

                    <!-- Summary -->
                    <div class="card bg-light mb-3">
                        <div class="card-body">
                            <div class="row g-2">
                                <div class="col-12">
                                    <div class="d-flex justify-content-between mb-2">
                                        <span class="text-muted">Subtotal:</span>
                                        <span class="fw-bold" id="subtotalDisplay">Rp 0</span>
                                    </div>
                                    <div class="d-flex justify-content-between mb-2" id="discountRow" style="display: none !important;">
                                        <span class="text-danger">Diskon:</span>
                                        <span class="text-danger fw-bold" id="discountDisplay">Rp 0</span>
                                    </div>
                                    <hr>
                                    <div class="d-flex justify-content-between">
                                        <span class="h5 mb-0">TOTAL:</span>
                                        <span class="h4 mb-0 text-primary fw-bold" id="totalDisplay">Rp 0</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Payment Section -->
                    <div class="row g-3 mb-3" id="paymentSection">
                        <div class="col-md-6">
                            <label for="payment_amount" class="form-label fw-bold">Jumlah Bayar</label>
                            <div class="input-group input-group-lg">
                                <span class="input-group-text">Rp</span>
                                <input type="number" class="form-control" id="payment_amount" 
                                       name="payment_amount" min="0" step="1000" required>
                            </div>
                            <div class="mt-2 d-flex gap-2 flex-wrap" id="quickAmountBtns">
                                <!-- Quick amount buttons will be added here -->
                            </div>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label fw-bold">Kembalian</label>
                            <div class="input-group input-group-lg">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control fw-bold" id="changeAmount" 
                                       value="0" readonly style="background: #e9ecef;">
                            </div>
                            <small class="text-muted" id="changeStatus"></small>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="notes" class="form-label">
                            <i class="fas fa-sticky-note me-1"></i>Catatan
                        </label>
                        <textarea class="form-control" id="notes" name="notes" rows="2" 
                                  placeholder="Catatan tambahan untuk transaksi..."></textarea>
                    </div>

                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-success btn-lg py-3 fw-bold" id="submitBtn" disabled>
                            <i class="fas fa-check-circle me-2"></i>PROSES TRANSAKSI
                        </button>
                        <button type="button" class="btn btn-outline-secondary" onclick="newTransaction()">
                            <i class="fas fa-redo me-2"></i>Transaksi Baru
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Success Modal -->
<div class="modal fade" id="successModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content border-0 shadow">
            <div class="modal-body text-center py-5">
                <div class="mb-4">
                    <i class="fas fa-check-circle text-success fa-5x"></i>
                </div>
                <h3 class="mb-3">Transaksi Berhasil!</h3>
                <div class="alert alert-success" id="successDetails"></div>
                <div class="d-grid gap-2">
                    <button type="button" class="btn btn-primary btn-lg" onclick="printReceipt()">
                        <i class="fas fa-print me-2"></i>Cetak Struk
                    </button>
                    <button type="button" class="btn btn-outline-primary" onclick="newTransaction()">
                        <i class="fas fa-plus me-2"></i>Transaksi Baru
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Customer Modal -->
<!-- Add Customer Modal -->
<div class="modal fade" id="addCustomerModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="fas fa-user-plus me-2"></i>Tambah Pelanggan Baru</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="addCustomerForm">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Nama Lengkap <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="newCustomerName" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Nomor Telepon <span class="text-danger">*</span></label>
                            <input type="tel" class="form-control" id="newCustomerPhone" required>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Email</label>
                        <input type="email" class="form-control" id="newCustomerEmail">
                    </div>
                   <!-- <div class="mb-3">
                        <label class="form-label">Alamat</label>
                        <textarea class="form-control" id="newCustomerAddress" rows="2"></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Tanggal Lahir</label>
                            <input type="date" class="form-control" id="newCustomerBirthDate">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Tipe Pelanggan</label>
                            <select class="form-select" id="newCustomerType">
                                <option value="regular">Regular</option>
                                <option value="member">Member</option>
                                <option value="vip">VIP</option>
                            </select>
                        </div>
                    </div>-->
                    <div class="mb-3">
                        <label class="form-label">Catatan</label>
                        <textarea class="form-control" id="newCustomerNotes" rows="2"></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-primary" onclick="saveNewCustomer()">
                    <i class="fas fa-save me-2"></i>Simpan
                </button>
            </div>
        </div>
    </div>
</div>

<style>
:root {
    --primary: #4361ee;
    --success: #06d6a0;
    --danger: #ef476f;
    --warning: #ffd60a;
}

.bg-gradient-primary {
    background: linear-gradient(135deg, var(--primary), #3730a3) !important;
}

.bg-gradient-success {
    background: linear-gradient(135deg, var(--success), #059669) !important;
}

.quantity-controls {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    background: white;
    border-radius: 50px;
    padding: 0.25rem;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.quantity-btn {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    background: #f8f9fa;
    transition: all 0.2s ease;
    cursor: pointer;
    font-size: 0.875rem;
}

.quantity-btn:hover:not(:disabled) {
    background: var(--primary);
    color: white;
    transform: scale(1.1);
}

.quantity-btn:disabled {
    opacity: 0.4;
    cursor: not-allowed;
}

.cart-item {
    background: white;
    border-radius: 12px;
    padding: 1rem;
    margin-bottom: 0.75rem;
    border: 2px solid #f1f3f5;
    transition: all 0.3s ease;
}

.cart-item:hover {
    border-color: var(--primary);
    box-shadow: 0 4px 12px rgba(67, 97, 238, 0.1);
    transform: translateY(-2px);
}

.product-card {
    border: 2px solid #e9ecef;
    border-radius: 12px;
    padding: 1rem;
    margin-bottom: 0.75rem;
    transition: all 0.3s ease;
    cursor: pointer;
    background: white;
}

.product-card:hover {
    border-color: var(--primary);
    box-shadow: 0 4px 16px rgba(67, 97, 238, 0.15);
    transform: translateX(4px);
}

.product-card:active {
    transform: scale(0.98);
}

.loading-spinner {
    display: inline-block;
    width: 20px;
    height: 20px;
    border: 3px solid #f3f3f3;
    border-top: 3px solid var(--primary);
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.quick-amount-btn {
    flex: 1;
    min-width: 70px;
    padding: 0.5rem;
    font-size: 0.875rem;
    border-radius: 8px;
    transition: all 0.2s ease;
}

.quick-amount-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

#cartContainer::-webkit-scrollbar,
#productList::-webkit-scrollbar {
    width: 8px;
}

#cartContainer::-webkit-scrollbar-track,
#productList::-webkit-scrollbar-track {
    background: #f1f3f5;
    border-radius: 10px;
}

#cartContainer::-webkit-scrollbar-thumb,
#productList::-webkit-scrollbar-thumb {
    background: #adb5bd;
    border-radius: 10px;
}

#cartContainer::-webkit-scrollbar-thumb:hover,
#productList::-webkit-scrollbar-thumb:hover {
    background: #868e96;
}

.badge-stock {
    font-size: 0.75rem;
    padding: 0.35em 0.65em;
    font-weight: 600;
}

.badge-stock.low {
    background-color: #ff6b6b !important;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.7; }
}

@media print {
    body * {
        visibility: hidden;
    }
    .receipt-content, .receipt-content * {
        visibility: visible;
    }
    .receipt-content {
        position: absolute;
        left: 0;
        top: 0;
    }
}
</style>

<script>
let cart = [];
let currentTransactionCode = "<?php echo $transaction_code; ?>";
let categories = [];
let allProducts = [];
let discountAmount = 0;

document.addEventListener('DOMContentLoaded', function() {
    loadProducts();
    updateCartDisplay();
    initEventListeners();
    updateClock();
    setInterval(updateClock, 1000);
    
    // MODIFIKASI: Tampilkan modal success jika ada data
    <?php if ($success_data): ?>
    setTimeout(() => {
        showSuccessModal(<?php echo json_encode($success_data); ?>);
    }, 500);
    <?php endif; ?>
});

function updateClock() {
    const now = new Date();
    const timeString = now.toLocaleTimeString('id-ID', { 
        hour: '2-digit', 
        minute: '2-digit',
        second: '2-digit'
    });
    const dateString = now.toLocaleDateString('id-ID', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
    document.getElementById('currentTime').textContent = timeString;
    document.getElementById('currentTime').title = dateString;
}

function initEventListeners() {
    document.getElementById('payment_amount').addEventListener('input', function() {
        calculateChange();
        validatePayment();
    });
    
    document.getElementById('payment_method').addEventListener('change', function() {
        updatePaymentSection();
    });
    
    document.getElementById('transactionForm').addEventListener('submit', function(e) {
        e.preventDefault();
        processTransaction();
    });
    
    document.getElementById('searchProduct').addEventListener('input', function(e) {
        filterProducts();
    });
    
    document.getElementById('categoryFilter').addEventListener('change', function() {
        filterProducts();
    });
    
    // Keyboard shortcuts
    document.addEventListener('keydown', function(e) {
        // F2 - Focus search
        if (e.key === 'F2') {
            e.preventDefault();
            document.getElementById('searchProduct').focus();
        }
        // F4 - Focus payment
        if (e.key === 'F4' && cart.length > 0) {
            e.preventDefault();
            document.getElementById('payment_amount').focus();
        }
        // F9 - Process transaction
        if (e.key === 'F9' && !document.getElementById('submitBtn').disabled) {
            e.preventDefault();
            processTransaction();
        }
    });
}

function updatePaymentSection() {
    const method = document.getElementById('payment_method').value;
    const paymentAmount = document.getElementById('payment_amount');
    const changeSection = paymentAmount.closest('.row').querySelector('.col-md-6:last-child');
    
    if (method === 'cash') {
        changeSection.style.display = 'block';
        paymentAmount.required = true;
        generateQuickAmounts();
    } else {
        changeSection.style.display = 'none';
        paymentAmount.value = calculateTotal();
        paymentAmount.required = false;
        document.getElementById('quickAmountBtns').innerHTML = '';
        calculateChange();
    }
}

function generateQuickAmounts() {
    const total = calculateTotal();
    const amounts = [
        Math.ceil(total / 1000) * 1000,
        Math.ceil(total / 5000) * 5000,
        Math.ceil(total / 10000) * 10000,
        Math.ceil(total / 50000) * 50000
    ];
    
    const uniqueAmounts = [...new Set(amounts)].filter(a => a >= total).slice(0, 4);
    
    const container = document.getElementById('quickAmountBtns');
    container.innerHTML = uniqueAmounts.map(amount => 
        `<button type="button" class="btn btn-sm btn-outline-primary quick-amount-btn" 
                onclick="setPaymentAmount(${amount})">
            ${formatNumber(amount)}
        </button>`
    ).join('');
}

function setPaymentAmount(amount) {
    document.getElementById('payment_amount').value = amount;
    calculateChange();
    validatePayment();
}

function loadProducts() {
    fetch('../api/get_products.php')
        .then(response => response.json())
        .then(products => {
            allProducts = products;
            extractCategories(products);
            displayProducts(products);
        })
        .catch(error => {
            console.error('Error:', error);
            showError('Gagal memuat produk');
        });
}

function extractCategories(products) {
    const uniqueCategories = [...new Set(products.map(p => p.category_name).filter(Boolean))];
    const categoryFilter = document.getElementById('categoryFilter');
    categoryFilter.innerHTML = '<option value="">Semua Kategori</option>' +
        uniqueCategories.map(cat => `<option value="${cat}">${cat}</option>`).join('');
}

function filterProducts() {
    const searchTerm = document.getElementById('searchProduct').value.toLowerCase();
    const category = document.getElementById('categoryFilter').value;
    
    const filtered = allProducts.filter(product => {
        const matchSearch = product.name.toLowerCase().includes(searchTerm) ||
                          (product.barcode && product.barcode.includes(searchTerm));
        const matchCategory = !category || product.category_name === category;
        return matchSearch && matchCategory;
    });
    
    displayProducts(filtered);
}

function displayProducts(products) {
    const productList = document.getElementById('productList');
    productList.innerHTML = '';
    
    if (products.length === 0) {
        productList.innerHTML = `
            <div class="text-center text-muted py-4">
                <i class="fas fa-search fa-3x mb-3 opacity-25"></i>
                <p>Tidak ada produk ditemukan</p>
            </div>`;
        return;
    }
    
    products.forEach(product => {
        const isLowStock = product.type !== 'service' && product.stock <= 5;
        const isOutOfStock = product.type !== 'service' && product.stock === 0;
        
        const productDiv = document.createElement('div');
        productDiv.className = 'product-card';
        if (isOutOfStock) productDiv.style.opacity = '0.5';
        
        productDiv.innerHTML = `
            <div class="d-flex justify-content-between align-items-start">
                <div class="flex-grow-1">
                    <h6 class="mb-1 fw-bold">${product.name}</h6>
                    <small class="text-muted d-block mb-2">${product.category_name || 'Umum'}</small>
                    <div class="d-flex align-items-center gap-2 mb-2">
                        <span class="badge bg-${getTypeBadgeColor(product.type)}">${formatType(product.type)}</span>
                        ${product.type !== 'service' ? `
                            <span class="badge badge-stock ${isLowStock ? 'low' : 'bg-secondary'}">
                                ${isOutOfStock ? 'HABIS' : `Stok: ${product.stock}`}
                            </span>
                        ` : ''}
                    </div>
                    <strong class="text-primary fs-6">Rp ${formatNumber(product.selling_price)}</strong>
                </div>
                <button class="btn btn-primary btn-sm ms-2" 
                        onclick="addToCart(${product.id}, '${escapeString(product.name)}', ${product.selling_price}, '${product.type}', ${product.stock})"
                        ${isOutOfStock ? 'disabled' : ''}>
                    <i class="fas fa-plus"></i>
                </button>
            </div>
        `;
        productList.appendChild(productDiv);
    });
}

function addToCart(productId, productName, price, type, stock) {
    if (type !== 'service' && stock <= 0) {
        showError('Stok habis!');
        return;
    }

    const existingItem = cart.find(item => item.productId === productId);
    
    if (existingItem) {
        if (type !== 'service') {
            if (existingItem.quantity >= stock) {
                showError('Stok tidak mencukupi!');
                return;
            }
            existingItem.quantity++;
        }
        existingItem.subtotal = existingItem.quantity * existingItem.price;
    } else {
        cart.push({
            productId: productId,
            name: productName,
            price: price,
            type: type,
            quantity: 1,
            subtotal: price
        });
    }
    
    updateCartDisplay();
    showSuccess(`${productName} ditambahkan ke keranjang`);
    
    // Play sound effect (optional)
    playSound('add');
}

function removeFromCart(index) {
    const item = cart[index];
    cart.splice(index, 1);
    updateCartDisplay();
    showSuccess(`${item.name} dihapus dari keranjang`);
    playSound('remove');
}

function updateQuantity(index, change) {
    const item = cart[index];
    
    if (item.type === 'service' && change > 0) {
        showError('Layanan tidak bisa ditambah kuantitas');
        return;
    }
    
    const newQuantity = item.quantity + change;
    
    if (newQuantity < 1) {
        removeFromCart(index);
        return;
    }
    
    item.quantity = newQuantity;
    item.subtotal = item.quantity * item.price;
    updateCartDisplay();
}

function clearCart() {
    if (cart.length === 0) return;
    
    if (confirm('Hapus semua item dari keranjang?')) {
        cart = [];
        discountAmount = 0;
        updateCartDisplay();
        document.getElementById('discountAmountInput').value = 0; // TAMBAHKAN INI
        showSuccess('Keranjang dikosongkan');
    }
}

function holdTransaction() {
    if (cart.length === 0) {
        showError('Keranjang kosong!');
        return;
    }
    
    const holdData = {
        cart: cart,
        discount: discountAmount,
        customer: document.getElementById('customer_id').value,
        timestamp: new Date().toISOString()
    };
    
    localStorage.setItem('hold_transaction', JSON.stringify(holdData));
    showSuccess('Transaksi ditunda. Klik "Ambil Transaksi" untuk melanjutkan');
    clearCart();
}

function retrieveHoldTransaction() {
    const holdData = localStorage.getItem('hold_transaction');
    if (!holdData) {
        showError('Tidak ada transaksi yang ditunda');
        return;
    }
    
    const data = JSON.parse(holdData);
    cart = data.cart;
    discountAmount = data.discount || 0;
    document.getElementById('customer_id').value = data.customer || '';
    
     updateCartDisplay();
    document.getElementById('discountAmountInput').value = discountAmount; // TAMBAHKAN INI
    localStorage.removeItem('hold_transaction');
    showSuccess('Transaksi berhasil diambil');
}

function applyDiscount() {
    if (cart.length === 0) {
        showError('Keranjang kosong!');
        return;
    }
    
    const currentTotal = calculateTotal();
    const discountInput = prompt('Masukkan diskon (Rp atau %):\nContoh: 10000 atau 10%', '0');
    
    if (!discountInput) return;
    
    let discount = 0;
    if (discountInput.includes('%')) {
        const percentage = parseFloat(discountInput.replace('%', ''));
        discount = (currentTotal * percentage) / 100;
    } else {
        discount = parseFloat(discountInput);
    }
    
    if (isNaN(discount) || discount < 0 || discount > currentTotal) {
        showError('Nilai diskon tidak valid!');
        return;
    }
    
    discountAmount = discount;
    updateCartDisplay();
    showSuccess(`Diskon Rp ${formatNumber(discount)} diterapkan`);
    
    // Update input hidden discount
    document.getElementById('discountAmountInput').value = discountAmount;
}

function updateCartDisplay() {
    const cartContainer = document.getElementById('cartContainer');
    const cartItemsInput = document.getElementById('cartItemsInput');
    const cartCount = document.getElementById('cartCount');
    const submitBtn = document.getElementById('submitBtn');
    
    const subtotal = cart.reduce((total, item) => total + item.subtotal, 0);
    const total = subtotal - discountAmount;
    
    cartCount.textContent = `${cart.length} items`;
    
    if (cart.length === 0) {
        cartContainer.innerHTML = `
            <div class="text-center text-muted py-5">
                <i class="fas fa-shopping-cart fa-4x mb-3 opacity-25"></i>
                <p class="mb-0">Keranjang kosong</p>
                <small>Tambahkan produk dari daftar di samping</small>
                ${localStorage.getItem('hold_transaction') ? 
                    '<button class="btn btn-sm btn-outline-primary mt-3" onclick="retrieveHoldTransaction()"><i class="fas fa-clock-rotate-left me-2"></i>Ambil Transaksi Tertunda</button>' 
                    : ''}
            </div>
        `;
        submitBtn.disabled = true;
    } else {
        cartContainer.innerHTML = cart.map((item, index) => `
            <div class="cart-item">
                <div class="d-flex justify-content-between align-items-start mb-2">
                    <div class="flex-grow-1">
                        <h6 class="mb-1 fw-bold">${item.name}</h6>
                        <small class="text-muted">${formatType(item.type)} • Rp ${formatNumber(item.price)}</small>
                    </div>
                    <button class="btn btn-sm btn-outline-danger" onclick="removeFromCart(${index})">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
                <div class="d-flex justify-content-between align-items-center">
                    <div class="quantity-controls">
                        <button class="quantity-btn" onclick="updateQuantity(${index}, -1)" 
                                ${item.type === 'service' ? 'disabled' : ''}>
                            <i class="fas fa-minus"></i>
                        </button>
                        <span class="fw-bold px-3">${item.quantity}</span>
                        <button class="quantity-btn" onclick="updateQuantity(${index}, 1)" 
                                ${item.type === 'service' ? 'disabled' : ''}>
                            <i class="fas fa-plus"></i>
                        </button>
                    </div>
                    <strong class="text-primary fs-5">Rp ${formatNumber(item.subtotal)}</strong>
                </div>
            </div>
        `).join('');
    }
    
    document.getElementById('discountAmountInput').value = discountAmount;
    document.getElementById('subtotalDisplay').textContent = `Rp ${formatNumber(subtotal)}`;
    document.getElementById('totalDisplay').textContent = `Rp ${formatNumber(total)}`;
    
    if (discountAmount > 0) {
        document.getElementById('discountRow').style.display = 'flex';
        document.getElementById('discountDisplay').textContent = `- Rp ${formatNumber(discountAmount)}`;
    } else {
        document.getElementById('discountRow').style.display = 'none';
    }
    
    cartItemsInput.value = JSON.stringify(cart);
    
    calculateChange();
    validatePayment();
    updatePaymentSection();
}

function calculateTotal() {
    const subtotal = cart.reduce((total, item) => total + item.subtotal, 0);
    return Math.max(0, subtotal - discountAmount);
}

function calculateChange() {
    const paymentMethod = document.getElementById('payment_method').value;
    const paymentAmount = parseFloat(document.getElementById('payment_amount').value) || 0;
    const total = calculateTotal();
    
    if (paymentMethod !== 'cash') {
        document.getElementById('changeAmount').value = '0';
        document.getElementById('changeStatus').textContent = '';
        return 0;
    }
    
    const change = paymentAmount - total;
    const changeAmountDisplay = document.getElementById('changeAmount');
    const changeStatus = document.getElementById('changeStatus');
    
    if (paymentAmount === 0) {
        changeAmountDisplay.value = '0';
        changeAmountDisplay.className = 'form-control fw-bold';
        changeStatus.textContent = '';
    } else if (change >= 0) {
        changeAmountDisplay.value = formatNumber(change);
        changeAmountDisplay.className = 'form-control fw-bold text-success';
        changeStatus.textContent = '✓ Pembayaran cukup';
        changeStatus.className = 'text-success';
    } else {
        changeAmountDisplay.value = formatNumber(Math.abs(change));
        changeAmountDisplay.className = 'form-control fw-bold text-danger';
        changeStatus.textContent = `✗ Kurang Rp ${formatNumber(Math.abs(change))}`;
        changeStatus.className = 'text-danger';
    }
    
    return change;
}

function validatePayment() {
    const paymentMethod = document.getElementById('payment_method').value;
    const paymentAmount = parseFloat(document.getElementById('payment_amount').value) || 0;
    const total = calculateTotal();
    const submitBtn = document.getElementById('submitBtn');
    
    let isValid = false;
    
    if (cart.length === 0) {
        isValid = false;
    } else if (paymentMethod === 'cash') {
        isValid = paymentAmount >= total;
    } else {
        isValid = true;
        document.getElementById('payment_amount').value = total;
    }
    
    submitBtn.disabled = !isValid;
    
    if (isValid) {
        submitBtn.className = 'btn btn-success btn-lg py-3 fw-bold';
        submitBtn.innerHTML = '<i class="fas fa-check-circle me-2"></i>PROSES TRANSAKSI (F9)';
    } else if (paymentAmount > 0 && paymentAmount < total) {
        submitBtn.className = 'btn btn-danger btn-lg py-3 fw-bold';
        submitBtn.innerHTML = `<i class="fas fa-exclamation-triangle me-2"></i>KURANG Rp ${formatNumber(total - paymentAmount)}`;
    } else {
        submitBtn.className = 'btn btn-secondary btn-lg py-3 fw-bold';
        submitBtn.innerHTML = '<i class="fas fa-check-circle me-2"></i>PROSES TRANSAKSI';
    }
}

function processTransaction() {
    const paymentMethod = document.getElementById('payment_method').value;
    const paymentAmount = parseFloat(document.getElementById('payment_amount').value) || 0;
    const total = calculateTotal();
    
    if (paymentMethod === 'cash' && paymentAmount < total) {
        showError(`Jumlah pembayaran kurang Rp ${formatNumber(total - paymentAmount)}`);
        return;
    }
    
    if (cart.length === 0) {
        showError('Keranjang belanja kosong!');
        return;
    }
    
    const submitBtn = document.getElementById('submitBtn');
    const originalHTML = submitBtn.innerHTML;
    
    submitBtn.innerHTML = '<span class="loading-spinner me-2"></span>Memproses...';
    submitBtn.disabled = true;
    
    // Submit form secara normal - TIDAK ADA REDIRECT MANUAL
    document.getElementById('transactionForm').submit();
}

// MODIFIKASI: Fungsi showSuccessModal - pastikan modal ditampilkan dengan benar
function showSuccessModal(data) {
    console.log('Showing success modal with data:', data); // Debug log
    
    // Store transaction data for printing
    window.lastTransaction = data;
    
    const modal = new bootstrap.Modal(document.getElementById('successModal'));
    const details = document.getElementById('successDetails');
    
    details.innerHTML = `
        <div class="text-start">
            <strong>Kode Transaksi:</strong> ${data.code}<br>
            <strong>Total Pembayaran:</strong> Rp ${formatNumber(data.total)}<br>
            <strong>Jumlah Bayar:</strong> Rp ${formatNumber(data.payment)}<br>
            <strong>Kembalian:</strong> Rp ${formatNumber(data.change)}
        </div>
    `;
    
    // Tampilkan modal
    modal.show();
    
    // Auto print receipt setelah modal tampil
    setTimeout(() => {
        printReceipt();
    }, 1000);
    
    playSound('success');
    
    // Reset cart setelah transaksi berhasil
    cart = [];
    discountAmount = 0;
    updateCartDisplay();
}


function printReceipt() {
    const data = window.lastTransaction;
    if (!data) {
        showError('Data transaksi tidak ditemukan');
        return;
    }
    
    const customerSelect = document.getElementById('customer_id');
    const customerName = customerSelect.options[customerSelect.selectedIndex]?.text.split(' - ')[0] || 'Walk-in Customer';
    const paymentMethod = document.getElementById('payment_method').value;
    
    const receiptContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <title>Struk Transaksi - ${data.code}</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { 
                    font-family: 'Courier New', monospace; 
                    font-size: 12px; 
                    padding: 10px;
                    width: 80mm;
                    margin: 0 auto;
                }
                .receipt-header { 
                    text-align: center; 
                    margin-bottom: 15px;
                    padding-bottom: 10px;
                    border-bottom: 2px dashed #000;
                }
                .receipt-header h2 { 
                    margin: 5px 0; 
                    font-size: 18px;
                    font-weight: bold;
                }
                .receipt-info { 
                    margin-bottom: 15px;
                    font-size: 11px;
                }
                .receipt-info div { margin: 3px 0; }
                .items-table { 
                    width: 100%; 
                    border-collapse: collapse;
                    margin: 15px 0;
                }
                .items-table th {
                    text-align: left;
                    padding: 5px 0;
                    border-bottom: 1px solid #000;
                    font-weight: bold;
                }
                .items-table td { 
                    padding: 5px 0;
                    vertical-align: top;
                }
                .items-table tr:not(:last-child) td {
                    border-bottom: 1px dashed #ccc;
                }
                .text-right { text-align: right; }
                .text-center { text-align: center; }
                .total-section { 
                    border-top: 2px solid #000;
                    margin-top: 15px;
                    padding-top: 10px;
                }
                .total-line {
                    display: flex;
                    justify-content: space-between;
                    margin: 5px 0;
                    font-size: 12px;
                }
                .total-line.grand {
                    font-size: 14px;
                    font-weight: bold;
                    margin-top: 10px;
                    padding-top: 10px;
                    border-top: 1px dashed #000;
                }
                .footer { 
                    margin-top: 20px;
                    padding-top: 10px;
                    border-top: 2px dashed #000;
                    text-align: center;
                    font-size: 11px;
                }
                .footer .barcode {
                    font-family: 'Libre Barcode 128', cursive;
                    font-size: 32px;
                    letter-spacing: 0;
                }
                @media print {
                    body { padding: 0; }
                    .no-print { display: none !important; }
                }
            </style>
        </head>
        <body>
            <div class="receipt-header">
                <h2>BGKL PRO SHOP</h2>
                <div>Jl. Contoh No. 123, Jakarta</div>
                <div>Telp: 0812-3456-7890</div>
                <div>Email: info@bgklproshop.com</div>
            </div>
            
            <div class="receipt-info">
                <div><strong>No. Transaksi:</strong> ${data.code}</div>
                <div><strong>Tanggal:</strong> ${new Date().toLocaleDateString('id-ID', {
                    weekday: 'long',
                    year: 'numeric',
                    month: 'long',
                    day: 'numeric'
                })}</div>
                <div><strong>Waktu:</strong> ${new Date().toLocaleTimeString('id-ID')}</div>
                <div><strong>Pelanggan:</strong> ${customerName}</div>
                <div><strong>Kasir:</strong> <?php echo $_SESSION['name']; ?></div>
            </div>
            
            <table class="items-table">
                <thead>
                    <tr>
                        <th style="width: 50%">ITEM</th>
                        <th style="width: 20%" class="text-center">QTY</th>
                        <th style="width: 30%" class="text-right">SUBTOTAL</th>
                    </tr>
                </thead>
                <tbody>
                    ${data.items.map(item => `
                        <tr>
                            <td>
                                ${item.name}<br>
                                <small>@ Rp ${formatNumber(item.price)}</small>
                            </td>
                            <td class="text-center">${item.quantity}</td>
                            <td class="text-right">Rp ${formatNumber(item.subtotal)}</td>
                        </tr>
                    `).join('')}
                </tbody>
            </table>
            
            <div class="total-section">
                <div class="total-line">
                    <span>Subtotal:</span>
                    <span>Rp ${formatNumber(data.total + (discountAmount || 0))}</span>
                </div>
                ${discountAmount > 0 ? `
                <div class="total-line">
                    <span>Diskon:</span>
                    <span>- Rp ${formatNumber(discountAmount)}</span>
                </div>
                ` : ''}
                <div class="total-line grand">
                    <span>TOTAL:</span>
                    <span>Rp ${formatNumber(data.total)}</span>
                </div>
                <div class="total-line">
                    <span>Bayar (${formatPaymentMethod(paymentMethod)}):</span>
                    <span>Rp ${formatNumber(data.payment)}</span>
                </div>
                <div class="total-line" style="font-weight: bold;">
                    <span>Kembali:</span>
                    <span>Rp ${formatNumber(data.change)}</span>
                </div>
            </div>
            
            <div class="footer">
                <div class="barcode">${data.code}</div>
                <div style="margin: 15px 0; font-size: 12px;">
                    <strong>*** TERIMA KASIH ***</strong>
                </div>
                <div>Barang yang sudah dibeli</div>
                <div>tidak dapat ditukar/dikembalikan</div>
                <div style="margin-top: 10px;">www.bgklproshop.com</div>
            </div>
            
            <div class="no-print" style="margin-top: 30px; text-align: center;">
                <button onclick="window.print()" style="padding: 12px 24px; background: #4361ee; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; margin-right: 10px;">
                    <span style="margin-right: 8px;">🖨️</span> Cetak Struk
                </button>
                <button onclick="window.close()" style="padding: 12px 24px; background: #6c757d; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 14px;">
                    <span style="margin-right: 8px;">✖️</span> Tutup
                </button>
            </div>
        </body>
        </html>
    `;
    
    const printWindow = window.open('', '_blank', 'width=400,height=700');
    printWindow.document.write(receiptContent);
    printWindow.document.close();
    
    setTimeout(() => {
        printWindow.print();
    }, 500);
}

function newTransaction() {
    if (cart.length > 0) {
        if (!confirm('Ada item dalam keranjang. Yakin ingin memulai transaksi baru?')) {
            return;
        }
    }
    window.location.href = 'transactions.php';
}

function saveNewCustomer() {
    const name = document.getElementById('newCustomerName').value.trim();
    const phone = document.getElementById('newCustomerPhone').value.trim();
    const email = document.getElementById('newCustomerEmail').value.trim();
    /*const address = document.getElementById('newCustomerAddress').value.trim();
    const birth_date = document.getElementById('newCustomerBirthDate').value;
    const customer_type = document.getElementById('newCustomerType').value;
    const notes = document.getElementById('newCustomerNotes').value.trim();*/

    if (!name || !phone) {
        showError('Nama dan nomor telepon harus diisi!');
        return;
    }

    // Tampilkan loading
    const saveBtn = document.querySelector('#addCustomerModal .btn-primary');
    const originalText = saveBtn.innerHTML;
    saveBtn.innerHTML = '<span class="loading-spinner me-2"></span>Menyimpan...';
    saveBtn.disabled = true;

    // Kirim data via AJAX
    const formData = new FormData();
    formData.append('name', name);
    formData.append('phone', phone);
    formData.append('email', email);
    /*formData.append('address', address);
    formData.append('birth_date', birth_date);
    formData.append('customer_type', customer_type);
    formData.append('notes', notes);*/

    fetch('../api/save_customer.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccess(data.message);
            
            // Tambahkan pelanggan baru ke dropdown
            const customerSelect = document.getElementById('customer_id');
            const optionText = `${data.customer.name} - ${data.customer.phone}`;
            const newOption = new Option(optionText, data.customer.id);
            customerSelect.appendChild(newOption);
            
            // Pilih pelanggan baru
            customerSelect.value = data.customer.id;

            // Tutup modal dan reset form
            bootstrap.Modal.getInstance(document.getElementById('addCustomerModal')).hide();
            document.getElementById('addCustomerForm').reset();
        } else {
            showError(data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showError('Terjadi kesalahan saat menyimpan pelanggan.');
    })
    .finally(() => {
        saveBtn.innerHTML = originalText;
        saveBtn.disabled = false;
    });
}

// Utility functions
function formatNumber(number) {
    return Number(number).toLocaleString('id-ID');
}

function escapeString(str) {
    return str.replace(/'/g, "\\'").replace(/"/g, '\\"').replace(/\n/g, '\\n');
}

function getTypeBadgeColor(type) {
    const colors = {
        'sparepart': 'primary',
        'service': 'success',
        'food': 'warning',
        'beverage': 'info'
    };
    return colors[type] || 'secondary';
}

function formatType(type) {
    const types = {
        'sparepart': 'Sparepart',
        'service': 'Layanan',
        'food': 'Makanan',
        'beverage': 'Minuman'
    };
    return types[type] || type;
}

function formatPaymentMethod(method) {
    const methods = {
        'cash': 'Tunai',
        'transfer': 'Transfer',
        'debit_card': 'Debit',
        'credit_card': 'Kredit',
        'e_wallet': 'E-Wallet'
    };
    return methods[method] || method;
}

function playSound(type) {
    // Optional: Add sound effects
    // You can use Web Audio API or HTML5 Audio
    try {
        const sounds = {
            'add': 'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBjGH0fPTgjMGHm7A7+OZSA0PVqzn77BdGAg+ltryxW8f',
            'remove': 'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBjGH0fPTgjMGHm7A7+OZSA0PVqzn77BdGAg+ltryxW8f',
            'success': 'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBjGH0fPTgjMGHm7A7+OZSA0PVqzn77BdGAg+ltryxW8f'
        };
        
        // Placeholder for sound implementation
        // const audio = new Audio(sounds[type]);
        // audio.play().catch(e => console.log('Sound play failed:', e));
    } catch (e) {
        console.log('Sound not supported');
    }
}

function showSuccess(message) {
    showNotification(message, 'success');
}

function showError(message) {
    showNotification(message, 'danger');
}

function showNotification(message, type = 'info') {
    const alertClass = `alert-${type}`;
    const icons = {
        'success': 'fa-check-circle',
        'danger': 'fa-exclamation-circle',
        'warning': 'fa-exclamation-triangle',
        'info': 'fa-info-circle'
    };
    
    const notification = document.createElement('div');
    notification.className = `alert ${alertClass} alert-dismissible fade show shadow-sm`;
    notification.style.cssText = 'position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px; animation: slideIn 0.3s ease;';
    notification.innerHTML = `
        <i class="fas ${icons[type]} me-2"></i>${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Add CSS animation
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
`;
document.head.appendChild(style);
</script>

<?php include '../includes/footer.php'; ?>