// assets/js/app.js

// Initialize POSApp only once
if (typeof window.POSApp === 'undefined') {
    window.POSApp = {
        // Notification System
        showNotification: function(message, type = 'info') {
            const alertClass = type === 'success' ? 'alert-success' : 
                              type === 'danger' ? 'alert-danger' : 
                              type === 'warning' ? 'alert-warning' : 'alert-info';
            
            const icons = {
                'success': 'fa-check-circle',
                'danger': 'fa-exclamation-circle',
                'warning': 'fa-exclamation-triangle',
                'info': 'fa-info-circle'
            };
            
            const notification = document.createElement('div');
            notification.className = `alert ${alertClass} alert-dismissible fade show shadow-sm`;
            notification.style.cssText = 'position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px; animation: slideInRight 0.3s ease;';
            notification.innerHTML = `
                <i class="fas ${icons[type]} me-2"></i>${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            document.body.appendChild(notification);
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.remove();
                    }
                }, 300);
            }, 5000);
        },

        // Format Currency
        formatCurrency: function(amount) {
            return new Intl.NumberFormat('id-ID', {
                style: 'currency',
                currency: 'IDR',
                minimumFractionDigits: 0
            }).format(amount);
        },

        // Debounce function for search
        debounce: function(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        },

        // Mobile Menu Toggle
        initMobileMenu: function() {
            const menuToggle = document.querySelector('.menu-toggle');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.mobile-overlay');

            if (menuToggle && sidebar) {
                menuToggle.addEventListener('click', () => {
                    sidebar.classList.toggle('mobile-open');
                    if (overlay) overlay.classList.toggle('active');
                });
            }

            // Close menu when clicking overlay
            if (overlay) {
                overlay.addEventListener('click', () => {
                    sidebar.classList.remove('mobile-open');
                    overlay.classList.remove('active');
                });
            }
        }
    };

    // Add CSS animation for notifications
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideInRight {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
    `;
    if (!document.querySelector('style[data-posapp-animations]')) {
        style.setAttribute('data-posapp-animations', 'true');
        document.head.appendChild(style);
    }
}

// Initialize app when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    if (window.POSApp && typeof window.POSApp.initMobileMenu === 'function') {
        window.POSApp.initMobileMenu();
    }
});

// Global helper functions
window.showSuccess = function(message) {
    if (window.POSApp && typeof window.POSApp.showNotification === 'function') {
        window.POSApp.showNotification(message, 'success');
    }
};

window.showError = function(message) {
    if (window.POSApp && typeof window.POSApp.showNotification === 'function') {
        window.POSApp.showNotification(message, 'danger');
    }
};

window.showWarning = function(message) {
    if (window.POSApp && typeof window.POSApp.showNotification === 'function') {
        window.POSApp.showNotification(message, 'warning');
    }
};

window.showInfo = function(message) {
    if (window.POSApp && typeof window.POSApp.showNotification === 'function') {
        window.POSApp.showNotification(message, 'info');
    }
};