<?php
namespace Reports\Controller;

use App\Controller\AppController;
use Cake\Datasource\ConnectionManager;
use DateTime;
use PHPExcel;
use PHPExcel_IOFactory;
use Setasign\Fpdf;

class ReportPenjualanPartsController extends AppController
{
    public $PhpExcel = null;

    public function index()
    {
        $conn = ConnectionManager::get('default');
        $data['titlePage'] = 'Report Penjualan Part';
        $data['titleMenu'] = 'Report Penjualan Part';

        if ($this->request->is('post')) {
            $start = $this->request->getData('start');
            $end = $this->request->getData('end');

            // Validasi input tanggal
            if (!$start || !$end || !strtotime($start) || !strtotime($end)) {
                $this->Flash->error(__('Tanggal tidak valid.'));
                return $this->redirect(['action' => 'index']);
            }

            try {
                // [1] QUERY DATA PENJUALAN PART - SIMPLE & CLEAN
                $result = $conn->execute("
                    SELECT 
                        a.trans_date, 
                        a.kode_workshop, 
                        a.nopol, 
                        COALESCE(e.nama_mekanik, '-') as nama_mekanik, 
                        COALESCE(e.employee_code, '-') as employee_code, 
                        c.partnumber, 
                        c.partdesc, 
                        c.qty as total_qty,
                        c.price, 
                        c.diskon, 
                        c.total_harga,
                        (c.price * c.qty * (1 - (c.diskon/100))) AS total_sales,
                        c.created
                    FROM workshops a 
                    INNER JOIN workshops_dtlparts c ON a.id = c.workshop_id
                    LEFT JOIN mst_employes e ON a.mekanik = e.employee_code
                    WHERE a.trans_date >= :start 
                        AND a.trans_date <= :end 
                        AND a.status != '1' 
                        AND c.status != '1'
                    ORDER BY a.trans_date ASC, a.kode_workshop ASC, c.id ASC
                ", ['start' => $start, 'end' => $end])->fetchAll('assoc');
                
                // Debug: Cek jumlah data untuk invoice tertentu
                $debugInv = array_filter($result, function($r) {
                    return $r['kode_workshop'] == 'INV251000362';
                });
                error_log("Total rows untuk INV251000362: " . count($debugInv));
                foreach ($debugInv as $d) {
                    error_log("  - Partnumber: {$d['partnumber']} | Desc: {$d['partdesc']} | Qty: {$d['total_qty']}");
                }

                // [2] QUERY PENDAPATAN JASA SERVICE
                $pendapatanJasa = $conn->execute("
                    SELECT 
                        e.employee_code, 
                        e.nama_mekanik, 
                        SUM(b.total_harga - b.diskon) AS jasa
                    FROM workshops a 
                    JOIN workshops_dtlservices b ON a.id = b.workshop_id
                    JOIN mst_employes e ON a.mekanik = e.employee_code
                    WHERE a.trans_date BETWEEN :start AND :end 
                        AND a.status != '1' 
                        AND b.status != '1'
                    GROUP BY e.employee_code, e.nama_mekanik
                ", ['start' => $start, 'end' => $end])->fetchAll('assoc');

                // [3] HITUNG HPP & PROFIT PART
                foreach ($result as $key => &$item) {
                    $partnumber = $item['partnumber'];
                    $transDate = $item['trans_date'];

                    // Query HPP dengan error handling
                    try {
                        $hppResult = $conn->execute("
                            SELECT h.harga_beli
                            FROM part_price_history h
                            WHERE h.partnumber = :partnumber
                              AND h.valid_from <= :trans_date
                              AND (h.valid_to IS NULL OR h.valid_to >= :trans_date)
                            ORDER BY h.valid_from DESC
                            LIMIT 1
                        ", ['partnumber' => $partnumber, 'trans_date' => $transDate])->fetch('assoc');

                        $hargaBeli = $hppResult['harga_beli'] ?? 0;
                    } catch (\Exception $e) {
                        // Jika error, set HPP = 0
                        $hargaBeli = 0;
                        error_log("Error getting HPP for {$partnumber}: " . $e->getMessage());
                    }

                    $item['total_hpp'] = $item['total_qty'] * $hargaBeli;
                    $item['profit'] = $item['total_sales'] - $item['total_hpp'];
                    
                    // Debug log untuk partnumber tertentu
                    if ($item['kode_workshop'] == 'INV251000362') {
                        error_log("Invoice: {$item['kode_workshop']} | Part: {$partnumber} | Qty: {$item['total_qty']} | HPP: {$hargaBeli} | Sales: {$item['total_sales']}");
                    }
                }

                // [4] HITUNG OMZET & PROFIT PER MEKANIK
                $profitPerMekanik = [];
                
                // --- A. PROSES DATA PART ---
                foreach ($result as $item) {
                    $empCode = $item['employee_code'];
                    
                    if (!isset($profitPerMekanik[$empCode])) {
                        $profitPerMekanik[$empCode] = [
                            'nama_mekanik' => $item['nama_mekanik'],
                            'omzet_part' => 0,
                            'profit_part' => 0,
                            'omzet_jasa' => 0,
                            'profit_jasa' => 0
                        ];
                    }
                    
                    $profitPerMekanik[$empCode]['omzet_part'] += $item['total_sales'];
                    $profitPerMekanik[$empCode]['profit_part'] += $item['profit'];
                }
                
                // --- B. PROSES DATA JASA ---
                foreach ($pendapatanJasa as $jasa) {
                    $empCode = $jasa['employee_code'];
                    
                    if (!isset($profitPerMekanik[$empCode])) {
                        $profitPerMekanik[$empCode] = [
                            'nama_mekanik' => $jasa['nama_mekanik'],
                            'omzet_part' => 0,
                            'profit_part' => 0,
                            'omzet_jasa' => 0,
                            'profit_jasa' => 0
                        ];
                    }
                    
                    // Asumsi profit jasa = omzet jasa (jika tidak ada biaya)
                    $profitPerMekanik[$empCode]['omzet_jasa'] += $jasa['jasa'];
                    $profitPerMekanik[$empCode]['profit_jasa'] += $jasa['jasa'];
                }
                
                // --- C. HITUNG TOTAL ---
                foreach ($profitPerMekanik as &$mekanik) {
                    $mekanik['total_omzet'] = $mekanik['omzet_part'] + $mekanik['omzet_jasa'];
                    $mekanik['total_profit'] = $mekanik['profit_part'] + $mekanik['profit_jasa'];
                }

                // [5] HITUNG TOTAL KESELURUHAN
                $totalOmzetPart = array_sum(array_column($result, 'total_sales'));
                $totalOmzetJasa = array_sum(array_column($pendapatanJasa, 'jasa'));
                $totalOmzet = $totalOmzetPart + $totalOmzetJasa;
                $totalProfitPart = array_sum(array_column($result, 'profit'));
                $totalProfitJasa = array_sum(array_column($pendapatanJasa, 'jasa'));
                $totalProfit = $totalProfitPart + $totalProfitJasa;

            } catch (\Exception $e) {
                $this->Flash->error(__('Terjadi kesalahan saat mengambil data: ' . $e->getMessage()));
                return $this->redirect(['action' => 'index']);
            }

            if (empty($result) && empty($pendapatanJasa)) {
                $this->Flash->set(__('Data tidak tersedia.'));
                return $this->redirect(['action' => 'index']);
            }

            $company = [
                'name' => 'AHASS RAJAWALI PUTERA',
                'address' => '-'
            ];

            $periode = date("d F Y", strtotime($start)) . ' - ' . date("d F Y", strtotime($end));
            
            $this->set(compact(
                'data', 
                'periode', 
                'company', 
                'result', 
                'profitPerMekanik',
                'totalOmzetPart',
                'totalOmzetJasa',
                'totalOmzet',
                'totalProfitPart', 
                'totalProfitJasa', 
                'totalProfit'
            ));

            switch ($this->request->getData('type')) {
                case 'excel':
                    $file = 'excel';
                    break;
                case 'pdf':
                    $file = 'pdf';
                    break;
                default:
                    $file = 'html';
                    break;
            }

            $this->render($file);
        }

        $this->set(compact('data'));
    }
}