<?php
declare(strict_types=1);

namespace Transactions\Controller;

use App\Controller\AppController;
use Cake\Datasource\ConnectionManager;
use Cake\ORM\TableRegistry;
use Mpdf\Mpdf;

class BillingInvoicesController extends AppController
{
    public function beforeFilter(\Cake\Event\EventInterface $event)
    {
        parent::beforeFilter($event);
        // Configure types untuk billing invoice
        $this->set('invoiceTypes', [
            'hosting' => 'Hosting',
            'maintenance' => 'Maintenance Server', 
            'software' => 'Penjualan Aplikasi',
            'domain' => 'Domain',
            'consulting' => 'Konsultasi IT',
            'other' => 'Lain-lain'
        ]);
    }

    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $listpayment = ['Cash', 'Transfer Bank', 'Credit Card', 'QRIS', 'Piutang'];
        $data['titlePage'] = 'Invoice Tagihan';
        $data['titleMenu'] = 'List Data Invoice Tagihan';

        $this->paginate = [
            'limit' => 20,
            'order' => [
                'created' => 'desc',
            ],
        ];

        $query = $this->BillingInvoices->find('all');

        if ($this->request->getQuery('search')) {
            $search = $this->request->getQuery('search');
            $query->where([
                'OR' => [
                    'invoice_number LIKE' => '%' . $search . '%',
                    'customer_name LIKE' => '%' . $search . '%',
                    'customer_company LIKE' => '%' . $search . '%',
                    'description LIKE' => '%' . $search . '%',
                ]
            ]);
        }

        // Filter by invoice type jika ada
        if ($this->request->getQuery('type')) {
            $type = $this->request->getQuery('type');
            $query->where(['invoice_type' => $type]);
        }

        $invoices = $this->paginate($query);

        $this->set(compact('data', 'invoices', 'listpayment'));
    }

    /**
     * View method
     *
     * @param string|null $id Invoice id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $listpayment = ['Cash', 'Transfer Bank', 'Credit Card', 'QRIS', 'Piutang'];
        
        $CompaniesTable = $this->fetchTable('SystemConfigurations.Companies');
        $companies = $CompaniesTable->find()->first();

        $data['titlePage'] = 'Invoice Tagihan';
        $data['titleMenu'] = 'View Data Invoice Tagihan';
        
        $invoice = $this->BillingInvoices->get($id, [
            'contain' => ['BillingInvoiceItems']
        ]);

        $this->set(compact('data', 'invoice', 'listpayment', 'companies'));
    }

    /**
     * Print method
     */
    public function prints($id = null)
    {
        $listpayment = ['Cash', 'Transfer Bank', 'Credit Card', 'QRIS', 'Piutang'];
        
        $CompaniesTable = $this->fetchTable('SystemConfigurations.Companies');
        $companies = $CompaniesTable->find()->first();

        $this->viewBuilder()->setLayout('print_css');
        $data['titlePage'] = 'Invoice Tagihan';
        $data['titleMenu'] = 'Cetak Invoice Tagihan';
        
        $invoice = $this->BillingInvoices->get($id, [
            'contain' => ['BillingInvoiceItems']
        ]);

        $this->set(compact('data', 'invoice', 'companies', 'listpayment'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $data['titlePage'] = 'Invoice Tagihan';
        $data['titleMenu'] = 'Tambah Invoice Tagihan';
        $this->loadComponent('General');
        $setStatus = 1;
        
        $invoice = $this->BillingInvoices->newEmptyEntity();
        
        if ($this->request->is('post')) {
            $connection = ConnectionManager::get('default');
            $connection->begin();
            
            $invoice = $this->BillingInvoices->patchEntity($invoice, $this->request->getData());

            // Generate invoice number
            $invoice->invoice_number = $this->General->__sinchronizeID('INV-BILL', date('ym'), 5, array('val_id','val_char', 'val_value'));
            $invoice->create_by = $this->getRequest()->getSession()->read('Auth.id');
            $invoice->modi_by = $this->getRequest()->getSession()->read('Auth.id');
            $invoice->invoice_date = date('Y-m-d');
            $invoice->due_date = $this->request->getData('due_date');
            
            // Calculate totals
            $invoice->subtotal = $this->General->__sinchronizecurrency($this->request->getData('subtotal'));
            $invoice->tax_amount = $this->General->__sinchronizecurrency($this->request->getData('tax_amount'));
            $invoice->discount_amount = $this->General->__sinchronizecurrency($this->request->getData('discount_amount'));
            $invoice->total_amount = $this->General->__sinchronizecurrency($this->request->getData('total_amount'));
            
            $invoice->branch_code = $this->getRequest()->getSession()->read('Auth.branch_code');
            $invoice->branch_id = $this->getRequest()->getSession()->read('Auth.branch_id');
            $invoice->status = 'draft';

            // Process invoice items
            if ($invoice->billing_invoice_items) {
                foreach ($invoice->billing_invoice_items as $i => $item) {
                    if (!empty($item['description'])) {
                        $invoice->billing_invoice_items[$i]->create_by = $this->getRequest()->getSession()->read('Auth.id');
                        $invoice->billing_invoice_items[$i]->modi_by = $this->getRequest()->getSession()->read('Auth.id');
                        $invoice->billing_invoice_items[$i]->created = date("Y-m-d H:i:s");
                        $invoice->billing_invoice_items[$i]->modified = date("Y-m-d H:i:s");
                        
                        $invoice->billing_invoice_items[$i]->unit_price = $this->General->__sinchronizecurrency($item['unit_price']);
                        $invoice->billing_invoice_items[$i]->quantity = $this->General->__sinchronizecurrency($item['quantity']);
                        $invoice->billing_invoice_items[$i]->total_price = $this->General->__sinchronizecurrency($item['total_price']);
                        
                        $invoice->billing_invoice_items[$i]->branch_code = $invoice->branch_code;
                        $invoice->billing_invoice_items[$i]->branch_id = $invoice->branch_id;
                    } else {
                        unset($invoice->billing_invoice_items[$i]);
                    }
                }
            }

            if ($setStatus) {
                if ($this->BillingInvoices->save($invoice, ['associated' => ['BillingInvoiceItems']])) {
                    $connection->commit();
                    $this->Flash->success(__('Invoice tagihan berhasil disimpan.'));
                    return $this->redirect(['action' => 'index']);
                } else {
                    $setStatus = 0;
                    $message = 'Invoice tagihan gagal disimpan. Silakan coba lagi.';
                }
            }

            if (!$setStatus) {
                $connection->rollback();
                $this->Flash->error(__($message));
            }
        }

        // Set default due date (30 hari dari sekarang)
        $defaultDueDate = date('Y-m-d', strtotime('+30 days'));
        
        $listpayment = ['Cash', 'Transfer Bank', 'Credit Card', 'QRIS', 'Piutang'];
        
        $this->set(compact('data', 'invoice', 'listpayment', 'defaultDueDate'));
    }

    /**
     * Edit method
     */
    public function edit($id = null)
    {
        $data['titlePage'] = 'Invoice Tagihan';
        $data['titleMenu'] = 'Edit Invoice Tagihan';
        $this->loadComponent('General');
        $setStatus = 1;
        
        $invoice = $this->BillingInvoices->get($id, [
            'contain' => ['BillingInvoiceItems']
        ]);

        if ($this->request->is(['put', 'post'])) {
            $connection = ConnectionManager::get('default');
            $connection->begin();
            
            $invoice = $this->BillingInvoices->patchEntity($invoice, $this->request->getData());
            $invoice->modi_by = $this->getRequest()->getSession()->read('Auth.id');
            $invoice->modified = date('Y-m-d H:i:s');

            // Calculate totals
            $invoice->subtotal = $this->General->__sinchronizecurrency($this->request->getData('subtotal'));
            $invoice->tax_amount = $this->General->__sinchronizecurrency($this->request->getData('tax_amount'));
            $invoice->discount_amount = $this->General->__sinchronizecurrency($this->request->getData('discount_amount'));
            $invoice->total_amount = $this->General->__sinchronizecurrency($this->request->getData('total_amount'));

            // Process invoice items
            if ($invoice->billing_invoice_items) {
                foreach ($invoice->billing_invoice_items as $i => $item) {
                    if (!empty($item['description'])) {
                        $invoice->billing_invoice_items[$i]->modi_by = $this->getRequest()->getSession()->read('Auth.id');
                        $invoice->billing_invoice_items[$i]->modified = date("Y-m-d H:i:s");
                        
                        $invoice->billing_invoice_items[$i]->unit_price = $this->General->__sinchronizecurrency($item['unit_price']);
                        $invoice->billing_invoice_items[$i]->quantity = $this->General->__sinchronizecurrency($item['quantity']);
                        $invoice->billing_invoice_items[$i]->total_price = $this->General->__sinchronizecurrency($item['total_price']);
                    } else {
                        unset($invoice->billing_invoice_items[$i]);
                    }
                }
            }

            if ($setStatus) {
                if ($this->BillingInvoices->save($invoice, ['associated' => ['BillingInvoiceItems']])) {
                    $connection->commit();
                    $this->Flash->success(__('Invoice tagihan berhasil diupdate.'));
                    return $this->redirect(['action' => 'index']);
                } else {
                    $setStatus = 0;
                    $message = 'Invoice tagihan gagal diupdate. Silakan coba lagi.';
                }
            }

            if (!$setStatus) {
                $connection->rollback();
                $this->Flash->error(__($message));
            }
        }

        $listpayment = ['Cash', 'Transfer Bank', 'Credit Card', 'QRIS', 'Piutang'];
        
        $this->set(compact('data', 'invoice', 'listpayment'));
    }

    /**
     * Delete method
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $invoice = $this->BillingInvoices->get($id);
        
        if ($this->BillingInvoices->delete($invoice)) {
            $this->Flash->success(__('Invoice tagihan berhasil dihapus.'));
        } else {
            $this->Flash->error(__('Invoice tagihan gagal dihapus. Silakan coba lagi.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    /**
     * Mark as paid method
     */
    public function markAsPaid($id = null)
    {
        $this->request->allowMethod(['post']);
        $invoice = $this->BillingInvoices->get($id);
        
        $invoice->status = 'paid';
        $invoice->paid_date = date('Y-m-d H:i:s');
        $invoice->modi_by = $this->getRequest()->getSession()->read('Auth.id');
        
        if ($this->BillingInvoices->save($invoice)) {
            $this->Flash->success(__('Invoice telah ditandai sebagai lunas.'));
        } else {
            $this->Flash->error(__('Gagal menandai invoice sebagai lunas. Silakan coba lagi.'));
        }

        return $this->redirect(['action' => 'view', $id]);
    }

    /**
     * Print PDF method
     */
    public function printpdf($id = null)
    {
        $listpayment = ['Cash', 'Transfer Bank', 'Credit Card', 'QRIS', 'Piutang'];
        
        $CompaniesTable = $this->fetchTable('SystemConfigurations.Companies');
        $companies = $CompaniesTable->find()->first();

        $this->viewBuilder()->setLayout('print');
        $data['titlePage'] = 'Invoice Tagihan';
        $data['titleMenu'] = 'Cetak Invoice Tagihan';
        
        $invoice = $this->BillingInvoices->get($id, [
            'contain' => ['BillingInvoiceItems']
        ]);

        $this->set(compact('data', 'invoice', 'companies', 'listpayment'));

        // Render the view and extract the HTML content
        $response = $this->render('prints');
        $html = $response->getBody()->__toString();

        // Initialize mPDF
        $mpdf = new Mpdf([
            'format' => 'A4',
            'orientation' => 'P',
        ]);

        // Load HTML content
        $mpdf->WriteHTML($html);
        
        // Output the PDF
        $mpdf->Output("Invoice-Tagihan-{$invoice->invoice_number}.pdf", \Mpdf\Output\Destination::INLINE);
    }

    /**
     * Get customer data for autocomplete
     */
    public function getCustomers()
    {
        $this->autoRender = false;
        $keyword = $this->request->getQuery('term');

        $customersTable = $this->fetchTable('Masters.Customers');
        $customers = $customersTable->find()
            ->where([
                'OR' => [
                    'nama LIKE' => '%' . $keyword . '%',
                    'perusahaan LIKE' => '%' . $keyword . '%',
                    'email LIKE' => '%' . $keyword . '%'
                ]
            ])
            ->select(['id', 'nama', 'perusahaan', 'email', 'alamat', 'no_tlp'])
            ->limit(10)
            ->toArray();

        $result = [];
        foreach ($customers as $customer) {
            $label = $customer->perusahaan ? $customer->perusahaan . ' - ' . $customer->nama : $customer->nama;
            $result[] = [
                'id' => $customer->id,
                'label' => $label,
                'value' => $label,
                'data' => [
                    'name' => $customer->nama,
                    'company' => $customer->perusahaan,
                    'email' => $customer->email,
                    'address' => $customer->alamat,
                    'phone' => $customer->no_tlp
                ]
            ];
        }

        echo json_encode($result);
        exit;
    }
}