<?php
namespace App\Controller;

use App\Controller\AppController;
use Cake\Datasource\ConnectionManager;
use Cake\Event\EventInterface;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Cake\I18n\Time;

class ReportKartuStocksController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        $this->loadComponent('RequestHandler');
    }

    public function beforeFilter(EventInterface $event)
    {
        parent::beforeFilter($event);
        $this->Authentication->allowUnauthenticated(['index']);
    }

    public function index()
    {
        $this->set('title', 'Laporan Kartu Stok');

        $conn = ConnectionManager::get('default');
        
        if ($this->request->is('post')) {
            $company['name'] = $this->request->getSession()->read('Auth.company_name');
            $company['address'] = $this->request->getSession()->read('Auth.company_address');

            $periode = $this->request->getData('year').'-'.$this->request->getData('month');
            $start = date('Y-m-01 00:00:00', strtotime($periode.'-01'));
            $end = date('Y-m-t 23:59:59', strtotime($periode.'-01'));
            $trx_date = date('Y-m-01', strtotime($periode.'-01'));
            
            $partno = $this->request->getData('partnumber');
            $branchcode = $this->request->getSession()->read('Auth.branch_code');
            
            // Query untuk mendapatkan data kartu stok
            $result = $conn->execute("
                SELECT id, trx_date, trx_code, part_no, part_name, warehouse_code, 
                       branch_code, rak_code, qty, in_out,
                       CASE 
                         WHEN trx_code LIKE 'PRC%' THEN 'Pembelian'
                         WHEN trx_code LIKE 'INV%' THEN 'Penjualan'
                         WHEN trx_code LIKE 'ST%' THEN 'Stok Opname'
                         WHEN trx_code LIKE 'BATAL%' THEN 'Pembatalan'
                         WHEN trx_code LIKE 'PS%' THEN 'Pengeluaran Non-Sales'
                         ELSE 'Lainnya'
                       END AS transaction_type
                FROM (
                    SELECT UUID() as id, periode trx_date, 'SALDO_AWAL' trx_code, part_no, part_name, 
                           warehouse_code, branch_code, rak_code, qty, 'SALDO_AWAL' as in_out 
                    FROM saldo_stock_bulanans
                    WHERE periode = '{$trx_date}' AND part_no = '{$partno}' AND branch_code='{$branchcode}'
                    
                    UNION ALL 
                    
                    SELECT id, trx_date, trx_code, part_no, part_name, warehouse_code, 
                           branch_code, rak_code, qty, in_out 
                    FROM log_parts
                    WHERE in_out='I' AND trx_date BETWEEN '".$start."' AND '".$end."' 
                      AND part_no = '{$partno}' AND branch_code='{$branchcode}'
                    
                    UNION ALL
                    
                    SELECT id, trx_date, trx_code, part_no, part_name, warehouse_code, 
                           branch_code, rak_code, qty, in_out 
                    FROM log_parts
                    WHERE in_out='O' AND trx_date BETWEEN '".$start."' AND '".$end."' 
                      AND part_no = '{$partno}' AND branch_code='{$branchcode}'
                ) a 
                ORDER BY trx_date ASC
            ");
            
            $data = $result->fetchAll('assoc');
            
            if (empty($data)) {
                $this->Flash->error(__('Data tidak tersedia untuk kriteria yang dipilih.'));
                return $this->redirect(['action' => 'index']);
            }
            
            // Hitung saldo berjalan
            $saldo = 0;
            foreach ($data as $key => $row) {
                if ($row['in_out'] == 'SALDO_AWAL' || $row['in_out'] == 'I') {
                    $saldo += $row['qty'];
                } else {
                    $saldo -= $row['qty'];
                }
                $data[$key]['saldo'] = $saldo;
            }
            
            // Ambil informasi part
            $partInfo = $conn->execute("
                SELECT partnumber, description, unit, min_stock, max_stock 
                FROM parts 
                WHERE partnumber = '{$partno}' 
                LIMIT 1
            ")->fetch('assoc');
            
            $exportType = $this->request->getData('type');
            
            switch ($exportType) {
                case 'excel':
                    return $this->exportToExcel($data, $partInfo, $company, $periode);
                    break;
                case 'pdf':
                    return $this->exportToPDF($data, $partInfo, $company, $periode);
                    break;
                default:
                    $this->set(compact('data', 'partInfo', 'company', 'periode'));
                    break;
            }
        }
        
        // Ambil daftar part untuk dropdown
        $partsList = $conn->execute("
            SELECT partnumber, description 
            FROM parts 
            WHERE status = 'A' 
            ORDER BY partnumber
        ")->fetchAll('assoc');
        
        $this->set(compact('partsList'));
    }
    
    private function exportToExcel($data, $partInfo, $company, $periode)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set header
        $sheet->setCellValue('A1', $company['name']);
        $sheet->setCellValue('A2', $company['address']);
        $sheet->setCellValue('A3', 'LAPORAN KARTU STOK');
        $sheet->setCellValue('A4', 'Periode: ' . date('F Y', strtotime($periode . '-01')));
        $sheet->setCellValue('A5', 'Part: ' . $partInfo['partnumber'] . ' - ' . $partInfo['description']);
        
        // Set column headers
        $sheet->setCellValue('A7', 'Tanggal');
        $sheet->setCellValue('B7', 'No. Transaksi');
        $sheet->setCellValue('C7', 'Jenis Transaksi');
        $sheet->setCellValue('D7', 'Keterangan');
        $sheet->setCellValue('E7', 'Masuk');
        $sheet->setCellValue('F7', 'Keluar');
        $sheet->setCellValue('G7', 'Saldo');
        $sheet->setCellValue('H7', 'Gudang');
        $sheet->setCellValue('I7', 'Rak');
        
        // Fill data
        $row = 8;
        foreach ($data as $item) {
            $sheet->setCellValue('A' . $row, date('d/m/Y', strtotime($item['trx_date'])));
            $sheet->setCellValue('B' . $row, $item['trx_code']);
            $sheet->setCellValue('C' . $row, $item['transaction_type']);
            $sheet->setCellValue('D' . $row, $this->getTransactionDescription($item['trx_code'], $item['transaction_type']));
            $sheet->setCellValue('E' . $row, ($item['in_out'] == 'I' || $item['in_out'] == 'SALDO_AWAL') ? $item['qty'] : 0);
            $sheet->setCellValue('F' . $row, $item['in_out'] == 'O' ? $item['qty'] : 0);
            $sheet->setCellValue('G' . $row, $item['saldo']);
            $sheet->setCellValue('H' . $row, $item['warehouse_code']);
            $sheet->setCellValue('I' . $row, $item['rak_code']);
            $row++;
        }
        
        // Style the header
        $headerStyle = [
            'font' => ['bold' => true],
            'alignment' => ['horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER],
            'fill' => ['fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID, 'startColor' => ['rgb' => 'E0E0E0']]
        ];
        $sheet->getStyle('A7:I7')->applyFromArray($headerStyle);
        
        // Auto size columns
        foreach (range('A', 'I') as $column) {
            $sheet->getColumnDimension($column)->setAutoSize(true);
        }
        
        // Set response headers
        $this->response = $this->response->withType('application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        $this->response = $this->response->withDownload('Kartu_Stok_' . $partInfo['partnumber'] . '_' . $periode . '.xlsx');
        
        $writer = new Xlsx($spreadsheet);
        $writer->save('php://output');
        
        return $this->response;
    }
    
    private function exportToPDF($data, $partInfo, $company, $periode)
    {
        // Implementation for PDF export would go here
        // This would typically use a PDF library like TCPDF or Dompdf
        
        // For now, we'll just redirect back with a message
        $this->Flash->info(__('Export PDF sedang dalam pengembangan.'));
        return $this->redirect(['action' => 'index']);
    }
    
    private function getTransactionDescription($trxCode, $transactionType)
    {
        $prefix = substr($trxCode, 0, 3);
        
        switch ($prefix) {
            case 'PRC':
                return 'Pembelian dari Supplier';
            case 'INV':
                return 'Penjualan ke Customer';
            case 'ST':
                return 'Stok Opname';
            case 'BAT':
                return 'Pembatalan Transaksi';
            case 'PS':
                return 'Pengeluaran Non-Sales';
            case 'SAL':
                return 'Saldo Awal Bulan';
            default:
                return $transactionType;
        }
    }
}